#!/bin/bash

# CAN bus dump script
# Usage: ./can_dump.sh [interface] [output_file]

# Default values
CAN_INTERFACE="can0"
OUTPUT_FILE="can_dump.txt"

# Function to display usage
usage() {
    echo "Usage: $0 [interface] [output_file]"
    echo "  interface: CAN interface name (default: can0)"
    echo "  output_file: Output file name (default: can_dump.txt)"
    echo "  Example: $0 can1 dump_2023.txt"
    exit 1
}

# Function to cleanup on exit
cleanup() {
    echo "Stopping CAN dump..."
    kill $PID 2>/dev/null
    exit 0
}

# Check if can-utils is installed
if ! command -v candump &> /dev/null; then
    echo "Error: candump not found. Please install can-utils package."
    echo "On Debian/Ubuntu: sudo apt-get install can-utils"
    echo "On RHEL/CentOS: sudo yum install can-utils"
    exit 1
fi

# Parse command line arguments
if [ $# -ge 1 ]; then
    CAN_INTERFACE=$1
fi

if [ $# -ge 2 ]; then
    OUTPUT_FILE=$2
fi

# Validate CAN interface exists
if ! ip link show $CAN_INTERFACE &> /dev/null; then
    echo "Error: CAN interface $CAN_INTERFACE not found"
    echo "Available interfaces:"
    ip link show | grep -E "can[0-9]" | awk -F': ' '{print $2}'
    exit 1
fi

# Set up signal handling
trap cleanup SIGINT SIGTERM

# Start dumping CAN messages
echo "Starting CAN dump on interface $CAN_INTERFACE"
echo "Output will be saved to $OUTPUT_FILE"
echo "Press Ctrl+C to stop"

# Create or clear output file
> "$OUTPUT_FILE"

# Log start time
echo "=== CAN Dump Started at $(date) ===" >> "$OUTPUT_FILE"
echo "Interface: $CAN_INTERFACE" >> "$OUTPUT_FILE"
echo "Output file: $OUTPUT_FILE" >> "$OUTPUT_FILE"
echo "===================================" >> "$OUTPUT_FILE"
echo "" >> "$OUTPUT_FILE"

# Start the dump process in background
candump -tz $CAN_INTERFACE > "$OUTPUT_FILE" 2>&1 &
PID=$!

# Wait for the process to complete
wait $PID

# Log completion
echo "" >> "$OUTPUT_FILE"
echo "=== CAN Dump Stopped at $(date) ===" >> "$OUTPUT_FILE"

echo "CAN dump completed. Output saved to $OUTPUT_FILE"